% Code for: Capital Reallocation and Aggregate Productivity
% R. Cooper & I. Schott
% Code version: September 2022

% Set last period to steady state
pit(T)      = pistar;
ct(T)       = cstar;

% In each time period, solve for k and pi jointly.
% First, assume no change in k, solve only for pi
% Then go through the optimal k choices, given those pi's etc.

for tt = T-1:-1:1


    t_left  = T - 1 - tt;

    % Consumption this period as a function of pi
    ct0         = @(x) x * At(tt) * (PARMS.meaneps * mu_at(tt) + phi_at(tt)) + (1-x)*At(tt)*(PARMS.meaneps*mu_nt(tt) + phi_nt(tt)) - x.^2/2*Ft(tt);
    % Derivative of current consumption w.r.t pi
    dc0         = @(x) 1/ct0(x) .* (At(tt)*( PARMS.meaneps*(mu_at(tt) - mu_nt(tt)) + phi_at(tt) - phi_nt(tt)) - x*Ft(tt));
    % Marginal effect coming from future periods before the steady state
    dc1         = 1./ct(tt+1:T-1) .* At(tt+1:T-1).*cumprod(1-pit(tt+1:T-1)).*(PARMS.meaneps*(mu_at(tt) - mu_nt(tt)) + PARMS.rho_eps.^transpose(1:t_left)*(phi_at(tt) - phi_nt(tt)));
    % Find the optimal pi
    pi_ss_foc   = At(end)/cstar*prod(1-pit(tt+1:T))*PARMS.rho_eps^(t_left+1)* phi_a*((1-PARMS.rho_eps)/(1 - PARMS.rho_eps*(1-pistar))) /(1 - PARMS.B*PARMS.rho_eps*(1-pistar));
    foc_pi0     = @(x) dc0(x) + PARMS.B.^(1:t_left)*dc1 + PARMS.B^(t_left+1)*pi_ss_foc;
    pit(tt)     = fzero(foc_pi0,pistar);

    % Collect other moments
    ct(tt)      = ct0(pit(tt));

end

%% Now given the series of pi_t, solve for optimal capital

for tt = T-1:-1:1

    t_left  = T - 1 - tt;

    % Derivative of current consumption with respect to capital (net of the k^(alpha - 1) term
    dc_dk0      = pit(tt)*PARMS.ALPHA/ct(tt)*At(tt).*(PARMS.eps*PARMS.rho_eps^0 + (1-PARMS.rho_eps^0)*PARMS.meaneps);
    % Derivative of stream of consumption before we are back to the steady state
    dc_dk1      = pit(tt)*PARMS.ALPHA./ct(tt+1:T-1)'.*At(tt+1:T-1)'.*cumprod(1-pit(tt+1:T-1))'.*(PARMS.eps*PARMS.rho_eps.^(1:t_left) + PARMS.meaneps*(1-PARMS.rho_eps.^(1:t_left)));
    % Derivative of stream of future consumption (once we are back to the steady state)
    k_ss_foc    = pit(tt)*PARMS.ALPHA/ct(tt)*At(T)*prod(1-pit(tt+1:T)).*(PARMS.rho_eps^(t_left + 1)*(PARMS.eps - PARMS.meaneps)/(1-PARMS.B*PARMS.rho_eps*(1-pistar)) + PARMS.meaneps/(1-PARMS.B*(1-pistar)) );

    zeta        = dc_dk0 + dc_dk1*PARMS.B.^(1:t_left)' + PARMS.B^(t_left+1)*k_ss_foc;
    lambda      = (PARMS.PP'* (zeta.^(1/(1-PARMS.ALPHA))) )^(1-PARMS.ALPHA);
    % Optimal k-vector given the guess for pi
    kt(tt,:)    = (zeta / lambda ).^(1/(1-PARMS.ALPHA));

end

% Collect endogenous moments
mu_at           = kt.^PARMS.ALPHA * PARMS.PP;
phi_at          = (PARMS.eps'.*kt.^PARMS.ALPHA)*PARMS.PP - PARMS.meaneps*mu_at;

% Update the series of mu_n and phi_n states, starting in the SS.
for tt = 1:T-1
    mu_nt(tt+1)     = pit(tt)*mu_at(tt) + (1-pit(tt))*mu_nt(tt); %#ok<SAGROW>
    phi_nt(tt+1)    = PARMS.rho_eps*(pit(tt)*phi_at(tt) + (1-pit(tt))*phi_nt(tt)); %#ok<SAGROW>
end

%% Now using this series of mu_at, phi_at, reoptimize pi until convergence
err     = 1;
while err > 1e-10

    pi_old  = pit;

    for tt = T-1:-1:1

        t_left  = T - 1 - tt;

        % Consumption this period as a function of pi
        ct0         = @(x) x * At(tt) * (PARMS.meaneps * mu_at(tt) + phi_at(tt)) + (1-x)*At(tt)*(PARMS.meaneps*mu_nt(tt) + phi_nt(tt)) - x.^2/2*Ft(tt);
        % Derivative of current consumption w.r.t pi
        dc0         = @(x) 1/ct0(x) .* (At(tt)*( PARMS.meaneps*(mu_at(tt) - mu_nt(tt)) + phi_at(tt) - phi_nt(tt)) - x*Ft(tt));
        % Marginal effect coming from future periods before the steady state
        dc1         = 1./ct(tt+1:T-1) .* At(tt+1:T-1).*cumprod(1-pit(tt+1:T-1)).*(PARMS.meaneps*(mu_at(tt) - mu_nt(tt)) + PARMS.rho_eps.^transpose(1:t_left)*(phi_at(tt) - phi_nt(tt)));
        % Find the optimal pi
        pi_ss_foc   = At(end)/cstar*prod(1-pit(tt+1:T))*PARMS.rho_eps^(t_left+1)* phi_a*((1-PARMS.rho_eps)/(1 - PARMS.rho_eps*(1-pistar))) /(1 - PARMS.B*PARMS.rho_eps*(1-pistar));
        foc_pi0     = @(x) dc0(x) + PARMS.B.^(1:t_left)*dc1 + PARMS.B^(t_left+1)*pi_ss_foc;
        pit(tt)     = fzero(foc_pi0,pistar);

        % Back out consumption
        ct(tt)      = ct0(pit(tt));

    end

    err     = max(pi_old - pit);

    for tt = T-1:-1:1

        t_left  = T - 1 - tt;

        % Derivative of current consumption with respect to capital (net of the k^(alpha - 1) term
        dc_dk0      = pit(tt)*PARMS.ALPHA/ct(tt)*At(tt).*(PARMS.eps*PARMS.rho_eps^0 + (1-PARMS.rho_eps^0)*PARMS.meaneps);
        % Derivative of stream of consumption before we are back to the steady state
        dc_dk1      = pit(tt)*PARMS.ALPHA./ct(tt+1:T-1)'.*At(tt+1:T-1)'.*cumprod(1-pit(tt+1:T-1))'.*(PARMS.eps*PARMS.rho_eps.^(1:t_left) + PARMS.meaneps*(1-PARMS.rho_eps.^(1:t_left)));
        % Derivative of stream of future consumption (once we are back to the steady state)
        k_ss_foc    = pit(tt)*PARMS.ALPHA/ct(tt)*At(T)*prod(1-pit(tt+1:T)).*(PARMS.rho_eps^(t_left + 1)*(PARMS.eps - PARMS.meaneps)/(1-PARMS.B*PARMS.rho_eps*(1-pistar)) + PARMS.meaneps/(1-PARMS.B*(1-pistar)) );
        zeta        = dc_dk0 + dc_dk1*PARMS.B.^(1:t_left)' + PARMS.B^(t_left+1)*k_ss_foc;
        lambda      = (PARMS.PP'* (zeta.^(1/(1-PARMS.ALPHA))) )^(1-PARMS.ALPHA);
        % Optimal k-vector given the guess for pi
        kt(tt,:)    = (zeta / lambda ).^(1/(1-PARMS.ALPHA));

    end

    % Collect endogenous moments
    mu_at           = kt.^PARMS.ALPHA * PARMS.PP;
    phi_at          = (PARMS.eps'.*kt.^PARMS.ALPHA)*PARMS.PP - PARMS.meaneps*mu_at;

end
%

for tt = 2:T
    mu_nt(tt)     = pit(tt-1)*mu_at(tt-1) + (1-pit(tt-1))*mu_nt(tt-1); %#ok<SAGROW>
    phi_nt(tt)    = PARMS.rho_eps*(pit(tt-1)*phi_at(tt-1) + (1-pit(tt-1))*phi_nt(tt-1)); %#ok<SAGROW>
end

% Add the initial, steady state period, delete last period
At      = [1; At(1:T-1)];
Ft      = [PARMS.F_upper; Ft(1:T-1)];
ct      = [cstar; ct(1:T-1)];
mu_at   = [mu_a; mu_at(1:T-1)];
mu_nt   = [mu_n; mu_nt(1:T-1)];
phi_at  = [phi_a; phi_at(1:T-1)];
phi_nt  = [phi_n; phi_nt(1:T-1)];
kt      = [steady.k_a'; kt(1:T-1,:)];
pit     = [pistar; pit(1:T-1)];

